import tkinter as tk
from tkinter import ttk, filedialog, messagebox
import cv2
from PIL import Image, ImageTk
import numpy as np
from ultralytics import YOLO
import time
import threading
import requests
import os
import torch
import platform
import subprocess
from pages.person_tracking import person_tracking_show
from pages.forgotten_objects import forgotten_objects_show
from pages.fall_detection import fall_detection_show
from pages.danger_zone import danger_zone_show
from pages.ServoKitCalib import ServoKit
np.bool = np.bool_ #Redefini np.bool pareil que np.bool_ pour compatibilité avec du code plus ancien (Utilisé pour tensorRT)

# CONFIGURATION
TELEGRAM_BOT_TOKEN = "#########"
TELEGRAM_CHAT_ID = "#######"
myServoKit = ######
myServoCam = 2
myLeftCam  = 4
myRightCam = 0
CamAssignAuto = True
# END CONFIGURATION

# État global de l'application
app_state = {
    "servo_camera": None,  # (path, cap) de la caméra sur le servomoteur
    "fix_cameras": {},     # Dictionnaire des caméras fixes {path: cap}
    "servo_kit": None      # Instance du servomoteur
}


def initialize_camera_position():
    """Initialise la position de la caméra pour qu'elle soit penchée vers la table"""
    try:
        kit = ServoKit(choixTK=myServoKit)
        # Position par défaut : pan = 0 (gauche), tilt = 60 (penché vers la table)
        kit.resetAll()
        time.sleep(2)
        kit.setAngle(1,  45)  # Tilt à 60 degrés pour viser la table
        time.sleep(2)  # Attendre que le servomoteur se positionne
        return True
    except Exception as e:
        print(f"Erreur lors de l'initialisation de la position de la caméra: {str(e)}")
        return False

#Fonction pour lister les caméras disponibles
def list_available_cameras():
    """Liste les caméras disponibles et leurs ports"""
    try:
        result = subprocess.run(['v4l2-ctl', '--list-devices'], capture_output=True, text=True)
        if result.stdout:
            camera_paths = []
            current_camera = None
            for line in result.stdout.split('\n'):
                if ':' in line and not '/dev/' in line:  # Nouvelle caméra physique
                    current_camera = line.strip().rstrip(':')  # Enlever les deux points à la fin
                elif '/dev/video' in line and current_camera:  # Premier port vidéo de la caméra
                    port = int(line.strip().split('video')[1])
                    camera_paths.append((current_camera, f'/dev/video{port}'))
                    current_camera = None  # Réinitialiser pour la prochaine caméra

            print("\n=== Caméras disponibles ===")
            for name, path in camera_paths:
                print(f"{name}: {path}")
            print("==========================\n")

            return camera_paths

    except Exception as e:
        print(f"Erreur lors de la liste des caméras: {str(e)}")
    return []



def move_camera(cap, port, kit):
    """Teste si une caméra est connectée au servomoteur"""
    try:
        if not cap.isOpened():
            return 0
        print("Capture image avant le mouvement")
        # Capture une image avant le mouvement
        ret, frame_before = cap.read()
        if not ret:
            return 0

        print(f"Rotation de la caméra {port} de 30 degrés")
        kit.setAngle(0, 30)  # Rotation de 30 degrés
        #time.sleep(1)  # Attendre que la caméra bouge complètement
        
        # Faire plusieurs captures et prendre la dernière pour s'assurer que la caméra a bien bougé
        for _ in range(5):
            ret, frame_after = cap.read()
            time.sleep(0.5)
        
        if not ret:
            return 0

        # Revenir à la position initiale
        kit.resetAll() # Retour à la position initiale
        time.sleep(2)  # Attendre que la caméra revienne

        # Calculer la différence entre les images
        # Convertir en niveaux de gris
        gray_before = cv2.cvtColor(frame_before, cv2.COLOR_BGR2GRAY)
        gray_after = cv2.cvtColor(frame_after, cv2.COLOR_BGR2GRAY)
        
        # Calculer la différence absolue
        diff = cv2.absdiff(gray_before, gray_after)
        
        # Calculer la moyenne des différences
        mean_diff = np.mean(diff)
        print(f"Différence moyenne pour la caméra {port}: {mean_diff:.2f}")
        
        # Sauvegarder les images pour debug
        cv2.imwrite(f'{port}_frame_before.jpg', frame_before)
        cv2.imwrite(f'{port}_frame_after.jpg', frame_after)
        cv2.imwrite(f'{port}_diff.jpg', diff)
        
        return mean_diff

    except Exception as e:
        print(f"Erreur lors du test de la caméra {port}: {str(e)}")
        return 0


#Fonction pour créer le stream d'une caméra
def create_camera_stream(camId):

    # Ouvrir la caméra
    cap = cv2.VideoCapture(camId, cv2.CAP_V4L2)
    if not cap.isOpened():
        print(f"Échec V4L2, tentative sans V4L2...")
        cap = cv2.VideoCapture(camId)

    if cap.isOpened():
        # Configuration de la caméra
        cap.set(cv2.CAP_PROP_FOURCC, cv2.VideoWriter_fourcc('M', 'J', 'P', 'G'))
        cap.set(cv2.CAP_PROP_FRAME_WIDTH, 1920)
        cap.set(cv2.CAP_PROP_FRAME_HEIGHT, 1080)
        cap.set(cv2.CAP_PROP_FPS, 30)
        cap.set(cv2.CAP_PROP_BUFFERSIZE, 1)

        return cap

#Fonction pour initialiser les caméras en dur via configuration
def initialize_cameras_hard():
    try:
        # Liste des caméras disponibles
        camera_paths = list_available_cameras()
        if not camera_paths:
            messagebox.showerror("Erreur", "Aucune caméra détectée")
            return False

        # Initialiser les caméras
        app_state["servo_camera"] = None
        app_state["fix_cameras"] = {}
        app_state["servo_kit"] = None
        # 'USB 2.0 Camera: HD USB Camera (usb-3610000.xhci-2.2)'

        # Ouvrir la caméra
        cap = create_camera_stream(myServoCam)
        path = ('USB 2.0 Camera: HD USB Camera (usb-3610000.xhci-2.2)', "/dev/video" + str(myServoCam))
        app_state["servo_camera"] = (path, cap)

        cap = create_camera_stream(myLeftCam)
        path = ('USB 2.0 Camera: HD USB Camera (usb-3610000.xhci-2.2)', "/dev/video"+str(myLeftCam))
        app_state["fix_cameras"][path] = (path, cap)

        cap = create_camera_stream(myRightCam)
        path = ('USB 2.0 Camera: HD USB Camera (usb-3610000.xhci-2.2)', "/dev/video"+str(myRightCam))
        app_state["fix_cameras"][path] = (path, cap)

        app_state["servo_kit"] = ServoKit(choixTK=myServoKit)

        # Libérer toutes les caméras existantes
        cv2.destroyAllWindows()

        return True

    except Exception as e:
        print(f"Erreur lors du test de la caméra {path}: {str(e)}")


#Fonction pour initialiser les caméras automatiquement
def initialize_cameras():
    """Initialise toutes les caméras au démarrage de l'application"""
    try:
        try:
            kit = ServoKit(choixTK=myServoKit)

            

        except Exception as e:
            print(f"Erreur d'init ServoKit")
            return 0

        # Liste des caméras disponibles
        camera_paths = list_available_cameras()

        if not camera_paths:
            messagebox.showerror("Erreur", "Aucune caméra détectée")
            return False

        # Initialiser les caméras
        app_state["servo_camera"] = None
        app_state["fix_cameras"] = {}
        app_state["servo_kit"] = None

        # Libérer toutes les caméras existantes
        for path in camera_paths:
            try:
                cap = cv2.VideoCapture(path[1])
                if cap.isOpened():
                    cap.release()
            except Exception as e:
                print(f"Erreur lors de la libération de {path}: {str(e)}")

        cv2.destroyAllWindows()
        time.sleep(1)  # Attendre que les caméras soient bien libérées

        print("Test des caméras pour trouver celle sur le servomoteur...")
        camera_results = []  # Liste pour stocker les résultats (path, cap, différence)

        # Tester chaque caméra
        for path in camera_paths[:3]:  # On ne prend que les 3 premières caméras
            try:
                port = int(path[1].split('video')[1])
                print(f"port: {port}")
                print(f"\nTest de la caméra {path}")

                # Ouvrir la caméra
                cap = cv2.VideoCapture(port, cv2.CAP_V4L2)
                if not cap.isOpened():
                    print(f"Échec V4L2, tentative sans V4L2...")
                    cap = cv2.VideoCapture(port)

                if cap.isOpened():
                    # Configuration de la caméra
                    cap.set(cv2.CAP_PROP_FOURCC, cv2.VideoWriter_fourcc('M', 'J', 'P', 'G'))
                    cap.set(cv2.CAP_PROP_FRAME_WIDTH, 1920)
                    cap.set(cv2.CAP_PROP_FRAME_HEIGHT, 1080)
                    cap.set(cv2.CAP_PROP_FPS, 30)
                    cap.set(cv2.CAP_PROP_BUFFERSIZE, 1)

                    # Tester le mouvement et obtenir la différence
                    diff = move_camera(cap, port, kit)
                    if diff > 0:  # Si la caméra a bougé
                        print(f"Caméra {path} a bougé avec une différence de {diff}")
                        camera_results.append((path, cap, diff))
                    else:
                        print(f"Caméra {path} n'a pas bougé")
                        cap.release()
                else:
                    print(f"Impossible d'ouvrir la caméra {path}")
            except Exception as e:
                print(f"Erreur lors du test de la caméra {path}: {str(e)}")
                continue

        # Trouver la caméra avec la plus grande différence
        if camera_results:
            # Trier par différence décroissante
            camera_results.sort(key=lambda x: x[2], reverse=True)
            best_path, best_cap, best_diff = camera_results[0]

            print(f"\nCaméra {best_path} avec {best_cap} a la plus grande différence de mouvement: {best_diff}")

            try:
                app_state["servo_kit"] = ServoKit(choixTK=myServoKit, NumCamera=int(best_path[1].split('video')[1]))
                app_state["servo_camera"] = (best_path, best_cap)  # Stocker le tuple (path, cap)

                app_state["servo_kit"].resetAll()
                time.sleep(2)

            except Exception as e:
                print(f"Erreur lors de l'initialisation du servomoteur: {str(e)}")
                best_cap.release()
                return False

            # Initialiser les autres caméras
            print("\n=== Initialisation des caméras fixes ===")
            for path, cap, diff in camera_results:  # Utiliser les captures déjà ouvertes
                print(f"\nTest caméra: {path}")
                print(f"best_path: {best_path}")
                if path[1] != best_path[1]:  # Si ce n'est pas la caméra sur le servomoteur
                    print(f"Initialisation de la caméra fixe: {path}")
                    try:
                        # Réutiliser la capture existante
                    
                        app_state["fix_cameras"][path] = (path, cap)
                        print(f"Caméra fixe {path} avec {cap} initialisée avec succès")
                    except Exception as e:
                        print(f"Erreur lors de l'initialisation de la caméra fixe {path}: {str(e)}")
                        continue
                else:
                    print(f"Caméra {path} est la caméra sur le servomoteur, ignorée")
            print("=== Fin initialisation des caméras fixes ===\n")

            return True
        else:
            messagebox.showerror("Erreur", "Aucune caméra sur servomoteur trouvée")
            return False

    except Exception as e:
        messagebox.showerror("Erreur", f"Erreur lors de l'initialisation: {str(e)}")
        return False


def send_telegram_alert(message):
    url = f"https://api.telegram.org/bot{TELEGRAM_BOT_TOKEN}/sendMessage"
    data = {"chat_id": TELEGRAM_CHAT_ID, "text": message}
    try:
        requests.post(url, data=data)
    except Exception as e:
        print(f"[Erreur Telegram] {e}")

def play_alert_sound():
    try:
        if platform.system() == "Windows":
            winsound.Beep(1000, 700)
        else:
            pygame.mixer.music.load("alert.wav")
            pygame.mixer.music.play()
    except Exception as e:
        print(f"[Erreur Audio] {e}")



def clear_frame(frame):
    for widget in frame.winfo_children():
        widget.destroy()

def show_main_menu(frame):
    clear_frame(frame)

    ttk.Label(frame, text="Système de Surveillance Intelligent", font=("Helvetica", 18, "bold")).pack(pady=10)

    # Frame pour les informations des caméras
    camera_info_frame = ttk.LabelFrame(frame, text="État des Caméras", padding=10)
    camera_info_frame.pack(pady=10, padx=10, fill="x")

    # Debug: Afficher le contenu de app_state
    print("\n=== État des caméras ===")
    print("Servo camera:", app_state["servo_camera"])
    print("Fix cameras:", app_state["fix_cameras"])
    print("=======================\n")

    # Afficher la caméra sur le servomoteur
    if app_state["servo_camera"]:
        servo_path = app_state["servo_camera"][0]  # On prend le path du tuple (path, cap)
        port = servo_path[1].split('video')[1]
        ttk.Label(camera_info_frame, text=f"Caméra sur servomoteur: /dev/video{port}").pack(anchor="w")

    # Afficher les autres caméras
    ttk.Label(camera_info_frame, text="Autres caméras:").pack(anchor="w", pady=(10,0))
    if not app_state["fix_cameras"]:
        ttk.Label(camera_info_frame, text="Aucune caméra fixe détectée").pack(anchor="w")
    else:
        for path, (camera_path, cap) in app_state["fix_cameras"].items():
            port = camera_path[1].split('video')[1]
            ttk.Label(camera_info_frame, text=f"Caméra fixe {port}: /dev/video{port}").pack(anchor="w")

    # Frame pour les boutons de contrôle
    control_frame = ttk.Frame(frame)
    control_frame.pack(pady=20)

    # Boutons de contrôle
    ttk.Button(control_frame, text="🚶 Tracking", width=20, command=lambda: show_tracking(frame, app_state)).grid(row=0, column=0, padx=10, pady=10)
    ttk.Button(control_frame, text="🎒 Objets", width=20, command=lambda: show_forgotten_objects(frame, app_state)).grid(row=0, column=1, padx=10, pady=10)
    ttk.Button(control_frame, text="🚨 Chute", width=20, command=lambda: show_fall_detection(frame, app_state)).grid(row=1, column=0, padx=10, pady=10)
    ttk.Button(control_frame, text="⚠️ Danger", width=20, command=lambda: show_danger_zone(frame, app_state)).grid(row=1, column=1, padx=10, pady=10)

    # Bouton de retour
    ttk.Button(frame, text="← Retour", command=lambda: frame.destroy()).pack(pady=10)


# === À compléter ===
def show_tracking(frame, app_state):
    clear_frame(frame)
    ttk.Label(frame, text="Suivi des personnes (Tracking)", font=("Helvetica", 16)).pack(pady=20)
    person_tracking_show(frame, app_state)
    ttk.Button(frame, text="← Retour au menu", command=lambda: show_main_menu(frame)).pack(pady=10)

def show_forgotten_objects(frame, app_state):
    clear_frame(frame)
    ttk.Label(frame, text="Détection d'objets oubliés", font=("Helvetica", 16)).pack(pady=20)
    initialize_camera_position()
    forgotten_objects_show(frame, app_state, root)
    ttk.Button(frame, text="← Retour au menu", command=lambda: show_main_menu(frame)).pack(pady=10)

def show_danger_zone(frame, app_state):
    clear_frame(frame)
    ttk.Label(frame, text="Détection de zone dangereuse", font=("Helvetica", 16)).pack(pady=20)
    danger_zone_show(frame, app_state)
    ttk.Button(frame, text="← Retour au menu", command=lambda: show_main_menu(frame)).pack(pady=10)

def show_fall_detection(frame, app_state):
    clear_frame(frame)
    ttk.Label(frame, text="Détection de Chutes", font=("Helvetica", 16)).pack(pady=20)
    initialize_camera_position()
    fall_detection_show(frame, app_state)
    ttk.Button(frame, text="← Retour au menu", command=lambda: show_main_menu(frame)).pack(pady=10)



def on_closing():
    # Libérer toutes les caméras
    if app_state["servo_camera"]:
        app_state["servo_camera"][1].release()
    for path, (camera_path, cap) in app_state["fix_cameras"].items():
        cap.release()
    cv2.destroyAllWindows()
    root.destroy()



if __name__ == "__main__":
    try:
        if platform.system() == "Windows":
            import winsound
        else:
            import pygame
            pygame.mixer.init()
    except:
        pass

    root = tk.Tk()
    root.title("Système de Surveillance Intelligent 🎥")
    # Obtenir les dimensions de l'écran
    screen_width = root.winfo_screenwidth()
    screen_height = root.winfo_screenheight()
    # Calculer 90% de la taille de l'écran
    window_width = int(screen_width * 0.9)
    window_height = int(screen_height * 0.9)
    # Calculer la position pour centrer la fenêtre
    x_position = (screen_width - window_width) // 2
    y_position = (screen_height - window_height) // 2
    # Configurer la fenêtre pour qu'elle prenne 90% de l'écran et soit centrée
    root.geometry(f"{window_width}x{window_height}+{x_position}+{y_position}")
    # Supprimer le mode plein écran
    # root.attributes('-fullscreen', True)

    style = ttk.Style()
    style.configure('TButton', font=('Helvetica', 12))
    main_frame = ttk.Frame(root, padding=20)
    main_frame.pack(expand=True, fill="both")



    # Initialiser les caméras au démarrage
    if CamAssignAuto:
        ret = initialize_cameras()
    else:
        ret = initialize_cameras_hard()
    if not ret:
        messagebox.showerror("Erreur", "Impossible d'initialiser les caméras. L'application va se fermer.")
        root.destroy()
        exit()

    # Lancer l'application
    show_main_menu(main_frame)

    # Fermer l'application
    root.protocol("WM_DELETE_WINDOW", on_closing)
    root.mainloop()